import cv2
import numpy as np
from pyzbar.pyzbar import decode
import requests
import os
from dotenv import load_dotenv
from urllib.parse import urlencode
import openai
from PIL import Image
import io
from tensorflow.keras.applications import ResNet50V2
from tensorflow.keras.preprocessing import image as tf_image

class AIEnhancedScanner:
    def __init__(self):
        self.model = ResNet50V2(weights='imagenet')
        
    def enhance_image(self, image_data):
        """Enhance image for better barcode detection"""
        nparr = np.frombuffer(image_data, np.uint8)
        img = cv2.imdecode(nparr, cv2.IMREAD_COLOR)
        
        # Image preprocessing
        gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
        denoised = cv2.fastNlMeansDenoising(gray)
        clahe = cv2.createCLAHE(clipLimit=2.0, tileGridSize=(8,8))
        enhanced = clahe.apply(denoised)
        
        return enhanced

    def scan_barcode(self, image_data):
        """Attempt barcode detection with enhanced preprocessing"""
        enhanced_img = self.enhance_image(image_data)
        barcodes = decode(enhanced_img)
        
        if barcodes:
            return barcodes[0].data.decode('utf-8')
        return None

    def recognize_image(self, image_data):
        """Backup image recognition if barcode fails"""
        img = tf_image.load_img(io.BytesIO(image_data), target_size=(224, 224))
        x = tf_image.img_to_array(img)
        x = np.expand_dims(x, axis=0)
        predictions = self.model.predict(x)
        return self._process_predictions(predictions)

    def _process_predictions(self, predictions):
        """Process model predictions into useful product information"""
        from tensorflow.keras.applications.resnet_v2 import decode_predictions
        return decode_predictions(predictions, top=5)[0]

class ProductDescriptionGenerator:
    def __init__(self):
        load_dotenv()
        openai.api_key = os.getenv('OPENAI_API_KEY')

    def generate_description(self, product_info):
        """Generate SEO-optimized product description"""
        prompt = f"""
        Create a detailed, SEO-optimized product listing for Facebook Marketplace with the following structure:

        Title: [Create an attention-grabbing title]
        Category: {product_info.get('category', 'General')}
        Condition: {product_info.get('condition', 'New')}
        
        Product Information:
        {product_info.get('details', '')}
        
        Please include:
        1. SEO-optimized title
        2. Detailed product description
        3. Key features and benefits
        4. Technical specifications
        5. Suggested keywords for search optimization
        """
        
        try:
            response = openai.chat.completions.create(
                model="gpt-3.5-turbo",
                messages=[
                    {"role": "system", "content": "You are a professional e-commerce product listing writer specializing in Facebook 
Marketplace listings."},
                    {"role": "user", "content": prompt}
                ]
            )
            return response.choices[0].message.content
        except Exception as e:
            return f"Error generating description: {str(e)}"

class FacebookIntegration:
    def __init__(self):
        load_dotenv()
        self.app_id = "1284867506091703"
        self.app_secret = "50eaaf9d314517df6500b5fd0d8c38a4"
        self.redirect_uri = "https://myjellyapp.host/facebook/auth/callback"
        self.scanner = AIEnhancedScanner()
        self.description_generator = ProductDescriptionGenerator()

    def get_login_url(self):
        params = {
            'client_id': self.app_id,
            'redirect_uri': self.redirect_uri,
            'scope': 'business_management,catalog_management,marketplace_product_management'
        }
        return f"https://www.facebook.com/v18.0/dialog/oauth?{urlencode(params)}"

    def get_access_token(self, code):
        params = {
            'client_id': self.app_id,
            'client_secret': self.app_secret,
            'redirect_uri': self.redirect_uri,
            'code': code
        }
        response = requests.get('https://graph.facebook.com/v18.0/oauth/access_token', params=params)
        return response.json() if response.ok else None

    def create_marketplace_listing(self, access_token, product_data, image_url=None):
        """Create a Facebook Marketplace listing"""
        try:
            # Create product catalog item
            catalog_data = {
                'name': product_data['title'],
                'description': product_data['description'],
                'price': product_data.get('price', 0),
                'currency': 'USD',
                'condition': product_data.get('condition', 'new'),
                'availability': 'in stock',
                'brand': product_data.get('brand', ''),
                'category': product_data.get('category', '')
            }
            
            if image_url:
                catalog_data['image_url'] = image_url

            # Create listing
            url = f"https://graph.facebook.com/v18.0/me/commerce_products"
            headers = {'Authorization': f'Bearer {access_token}'}
            response = requests.post(url, json=catalog_data, headers=headers)
            
            if response.ok:
                product_id = response.json().get('id')
                # Create marketplace listing
                listing_data = {
                    'product_id': product_id,
                    'delivery_method': 'shipping'
                }
                listing_response = requests.post(
                    f"https://graph.facebook.com/v18.0/me/marketplace_listings",
                    json=listing_data,
                    headers=headers
                )
                return listing_response.json() if listing_response.ok else None
            return None
        except Exception as e:
            return {'error': str(e)}

    def process_product_image(self, image_data):
        """Process product image with AI enhancement"""
        # Try barcode scanning first
        barcode = self.scanner.scan_barcode(image_data)
        if barcode:
            return {'type': 'barcode', 'code': barcode}
            
        # Fall back to image recognition
        recognition_results = self.scanner.recognize_image(image_data)
        return {'type': 'image_recognition', 'results': recognition_results}

    def generate_product_listing(self, image_data, basic_info=None):
        """Generate complete product listing with AI assistance"""
        # Process image for identification
        product_info = self.process_product_image(image_data)
        
        # Combine with basic info if provided
        if basic_info:
            product_info.update(basic_info)
        
        # Generate AI description
        description = self.description_generator.generate_description(product_info)
        
        return {
            'product_info': product_info,
            'description': description
        }
