#!/bin/bash

# Configuration
APP_PATH="/home/myjezdjr/public_html/facebook"
VENV_PATH="/home/myjezdjr/virtualenv/public_html/facebook/3.12"

echo "Starting setup..."

# Activate virtual environment
source $VENV_PATH/bin/activate

# Clean existing files
rm -rf $APP_PATH/migrations
rm -rf $APP_PATH/__pycache__
rm -rf $APP_PATH/app/__pycache__

# Install packages
pip install flask==2.0.1 flask-sqlalchemy==2.5.1 flask-migrate==3.1.0 sqlalchemy==1.4.23 python-dotenv pillow Werkzeug==2.0.3

# Create directories
mkdir -p $APP_PATH/app/templates
mkdir -p $APP_PATH/app/static
mkdir -p $APP_PATH/migrations
mkdir -p $APP_PATH/tmp

# Create __init__.py
cat > $APP_PATH/app/__init__.py << 'END'
from flask import Flask
from flask_sqlalchemy import SQLAlchemy
from flask_migrate import Migrate
from dotenv import load_dotenv
import os

db = SQLAlchemy()
migrate = Migrate()

def create_app():
    app = Flask(__name__)
    load_dotenv()
    
    app.config['SECRET_KEY'] = os.getenv('SECRET_KEY', 'default-secret-key')
    app.config['SQLALCHEMY_DATABASE_URI'] = os.getenv('DATABASE_URL', 'sqlite:///app.db')
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    
    db.init_app(app)
    migrate.init_app(app, db)
    
    from app.routes import bp as api_bp
    app.register_blueprint(api_bp)
    
    return app
END

# Create routes.py
cat > $APP_PATH/app/routes.py << 'END'
from flask import Blueprint, request, jsonify

bp = Blueprint('api', __name__)

@bp.route('/scan', methods=['POST'])
def scan_product():
    return jsonify({'status': 'Scan endpoint ready'})

@bp.route('/health', methods=['GET'])
def health_check():
    return jsonify({'status': 'healthy'})
END

# Create services.py
cat > $APP_PATH/app/services.py << 'END'
from PIL import Image
import io

class BarcodeScanner:
    def scan(self, image_data):
        return {'status': 'Barcode scanning ready'}

class ImageRecognition:
    def identify(self, image_data):
        return {'status': 'Image recognition ready'}

class FacebookIntegration:
    def create_listing(self, product_data):
        return {'status': 'Facebook integration ready'}
END

# Create wsgi.py
cat > $APP_PATH/wsgi.py << 'END'
from app import create_app

app = create_app()

if __name__ == "__main__":
    app.run()
END

# Create .env file
cat > $APP_PATH/.env << END
SECRET_KEY=your-secret-key-here
DATABASE_URL=sqlite:///app.db
FLASK_APP=wsgi.py
FLASK_ENV=development
END

# Set permissions
chmod -R 755 $APP_PATH
find $APP_PATH -type f -exec chmod 644 {} \;

# Initialize database
export FLASK_APP=$APP_PATH/wsgi.py
cd $APP_PATH
flask db init
flask db migrate -m "Initial migration"
flask db upgrade

# Restart application
touch $APP_PATH/tmp/restart.txt

echo "Setup complete!"
