
#!/bin/bash

# Configuration
APP_PATH="/home/myjezdjr/public_html/facebook"
VENV_PATH="/home/myjezdjr/virtualenv/public_html/facebook/3.12"
BACKUP_NAME="backup_before_setup_$(date +%Y%m%d_%H%M%S).tar.gz"

# Error handling
set -e  # Exit on any error
trap 'handle_error $? $LINENO' ERR

handle_error() {
    echo "Error occurred in script at line: $2"
    echo "Error code: $1"
    echo "Rolling back changes..."
    
    if [ -f "$APP_PATH/../$BACKUP_NAME" ]; then
        cd $APP_PATH/..
        rm -rf facebook
        tar -xzf $BACKUP_NAME
        echo "Restored from backup"
    fi
    
    exit 1
}

echo "Starting safe setup process..."

# Create backup first
echo "Creating backup..."
cd $APP_PATH/..
tar -czf $BACKUP_NAME facebook/ || {
    echo "Failed to create backup! Aborting."
    exit 1
}

# Validate virtual environment
if [ ! -f "$VENV_PATH/bin/activate" ]; then
    echo "Virtual environment not found! Aborting."
    exit 1
fi

# Activate virtual environment
source $VENV_PATH/bin/activate || {
    echo "Failed to activate virtual environment! Aborting."
    exit 1
}

# Clean existing installation
echo "Cleaning existing installation..."
rm -rf $APP_PATH/migrations
rm -rf $APP_PATH/__pycache__
rm -rf $APP_PATH/app/__pycache__

# Install packages with version checking
echo "Installing required packages..."
pip install --no-cache-dir \
    flask==2.0.1 \
    flask-sqlalchemy==2.5.1 \
    flask-migrate==3.1.0 \
    sqlalchemy==1.4.23 \
    python-dotenv \
    pillow \
    Werkzeug==2.0.3 || {
    echo "Package installation failed! Rolling back..."
    handle_error $? $LINENO
}

# Verify installations
python3 -c "import flask; import flask_sqlalchemy; import flask_migrate" || {
    echo "Package verification failed! Rolling back..."
    handle_error $? $LINENO
}

# Create directory structure
echo "Creating directory structure..."
mkdir -p $APP_PATH/app/templates
mkdir -p $APP_PATH/app/static
mkdir -p $APP_PATH/migrations
mkdir -p $APP_PATH/tmp

# Create application files
echo "Creating application files..."

# Create __init__.py
cat > $APP_PATH/app/__init__.py << 'EOF'
from flask import Flask
from flask_sqlalchemy import SQLAlchemy
from flask_migrate import Migrate
from dotenv import load_dotenv
import os

db = SQLAlchemy()
migrate = Migrate()

def create_app():
    app = Flask(__name__)
    load_dotenv()
    
    app.config['SECRET_KEY'] = os.getenv('SECRET_KEY', 'default-secret-key')
    app.config['SQLALCHEMY_DATABASE_URI'] = os.getenv('DATABASE_URL', 'sqlite:///app.db')
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    
    db.init_app(app)
    migrate.init_app(app, db)
    
    from app.routes import bp as api_bp
    app.register_blueprint(api_bp)
    
    @app.route('/test')
    def test():
        return {'status': 'working'}
    
    return app
EOF

# Create routes.py
cat > $APP_PATH/app/routes.py << 'EOF'
from flask import Blueprint, request, jsonify

bp = Blueprint('api', __name__)

@bp.route('/scan', methods=['POST'])
def scan_product():
    return jsonify({'status': 'Scan endpoint ready'})

@bp.route('/health', methods=['GET'])
def health_check():
    return jsonify({'status': 'healthy'})
EOF

# Create services.py
cat > $APP_PATH/app/services.py << 'EOF'
from PIL import Image
import io

class BarcodeScanner:
    def scan(self, image_data):
        return {'status': 'Barcode scanning ready'}

class ImageRecognition:
    def identify(self, image_data):
        return {'status': 'Image recognition ready'}

class FacebookIntegration:
    def create_listing(self, product_data):
        return {'status': 'Facebook integration ready'}
EOF

# Create wsgi.py
cat > $APP_PATH/wsgi.py << 'EOF'
from app import create_app

app = create_app()

if __name__ == "__main__":
    app.run()
EOF

# Create passenger_wsgi.py
cat > $APP_PATH/passenger_wsgi.py << 'EOF'
import sys
import os

INTERP = "/home/myjezdjr/virtualenv/public_html/facebook/3.12/bin/python"
if sys.executable != INTERP:
    os.execl(INTERP, INTERP, *sys.argv)

sys.path.append(os.getcwd())

from wsgi import app as application
EOF

# Create .env file
cat > $APP_PATH/.env << EOF
SECRET_KEY=$(python3 -c 'import secrets; print(secrets.token_hex(32))')
DATABASE_URL=sqlite:///app.db
FLASK_APP=wsgi.py
FLASK_ENV=development
EOF

# Set permissions
echo "Setting permissions..."
chmod -R 755 $APP_PATH
find $APP_PATH -type f -exec chmod 644 {} \;
chmod +x $APP_PATH/tmp

# Initialize database
echo "Initializing database..."
export FLASK_APP=$APP_PATH/wsgi.py
cd $APP_PATH
flask db init || {
    echo "Database initialization failed! Rolling back..."
    handle_error $? $LINENO
}

flask db migrate -m "Initial migration" || {
    echo "Database migration creation failed! Rolling back..."
    handle_error $? $LINENO
}

flask db upgrade || {
    echo "Database upgrade failed! Rolling back..."
    handle_error $? $LINENO
}

# Restart application
touch $APP_PATH/tmp/restart.txt

# Verify installation
echo "Verifying installation..."
curl -s http://localhost/facebook/test > /dev/null || {
    echo "Application verification failed! Please check your logs."
}

echo "Setup completed successfully!"
echo "Your application should now be accessible."
echo "Test URL: yourdomain.com/facebook/test"
echo "Health check: yourdomain.com/facebook/api/health"

# Save log of what we did
echo "Setup completed at $(date)" > $APP_PATH/setup_log.txt
