#!/bin/bash

# Configuration
APP_PATH="/home/myjezdjr/public_html/facebook"
VENV_PATH="/home/myjezdjr/virtualenv/public_html/facebook/3.12"

echo "Starting complete setup..."

# Error handling
set -e
trap 'echo "Error occurred in script at line: $LINENO"' ERR

# Create backup
echo "Creating backup..."
cd $APP_PATH/..
tar -czf facebook_backup_$(date +%Y%m%d_%H%M%S).tar.gz facebook/

# Clean existing installation
echo "Cleaning existing installation..."
cd $APP_PATH
rm -rf migrations __pycache__ app/__pycache__ *.pyc app/*.pyc

# Activate virtual environment
source $VENV_PATH/bin/activate

# Remove existing packages
echo "Removing existing packages..."
pip uninstall -y flask werkzeug flask-sqlalchemy flask-migrate

# Install specific versions
echo "Installing required packages..."
pip install werkzeug==2.0.3
pip install flask==2.0.1
pip install flask-sqlalchemy==2.5.1
pip install flask-migrate==3.1.0

# Create directory structure
echo "Creating directory structure..."
mkdir -p $APP_PATH/app/templates
mkdir -p $APP_PATH/app/static
mkdir -p $APP_PATH/migrations
mkdir -p $APP_PATH/tmp

# Create __init__.py
echo "Creating __init__.py..."
cat > $APP_PATH/app/__init__.py << 'EOF'
from flask import Flask

def create_app():
    app = Flask(__name__)
    
    @app.route('/')
    def home():
        return 'Home page - test'

    @app.route('/test')
    def test():
        return 'Test page'
    
    return app
EOF

# Create wsgi.py
echo "Creating wsgi.py..."
cat > $APP_PATH/wsgi.py << 'EOF'
from app import create_app

app = create_app()

if __name__ == "__main__":
    app.run()
EOF

# Create passenger_wsgi.py
echo "Creating passenger_wsgi.py..."
cat > $APP_PATH/passenger_wsgi.py << 'EOF'
import sys
import os

INTERP = "/home/myjezdjr/virtualenv/public_html/facebook/3.12/bin/python"
if sys.executable != INTERP:
    os.execl(INTERP, INTERP, *sys.argv)

sys.path.append(os.getcwd())

from wsgi import app as application
EOF

# Create .htaccess
echo "Creating .htaccess..."
cat > $APP_PATH/.htaccess << 'EOF'
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteRule ^(.*)$ public_html/facebook/wsgi.py/$1 [QSA,L]
EOF

# Set correct permissions
echo "Setting permissions..."
chmod -R 755 $APP_PATH
find $APP_PATH -type f -exec chmod 644 {} \;
chmod +x $APP_PATH/tmp
chmod +x $APP_PATH/app/static
chmod +x $APP_PATH/app/templates

# Verify Flask installation
echo "Verifying Flask installation..."
python3 -c "from flask import Flask; print('Flask installation verified')"

# Initialize database
echo "Initializing database..."
export FLASK_APP=$APP_PATH/wsgi.py
flask db init || echo "Database initialization failed - continuing anyway"

# Create test database
echo "Creating test database..."
python3 - << 'EOF'
from app import create_app
from flask_sqlalchemy import SQLAlchemy

app = create_app()
db = SQLAlchemy(app)
with app.app_context():
    db.create_all()
EOF

# Restart application
echo "Restarting application..."
touch $APP_PATH/tmp/restart.txt

echo "Setup complete!"
echo "Testing Flask import..."
python3 -c "from flask import Flask; print('Flask import successful')"
echo "You can now access your application at: https://myjellyapp.host/facebook"
